class BlogsController < ApplicationController
  unloadable

  before_filter :find_blog, :except => [:new, :index, :preview]
  before_filter :find_project, :only => [:new, :preview]
  before_filter :authorize, :except => [:index, :preview]
  before_filter :find_optional_project, :only => :index
  accept_key_auth :index

  def index
    @blogs_pages, @blogs = paginate :blogs,
      :per_page => 10,
      :conditions => (@project ? {:project_id => @project.id} : Project.visible_by(User.current)),
      :include => [:author, :project],
      :order => "#{Blog.table_name}.created_on DESC"
    respond_to do |format|
      format.html { render :layout => false if request.xhr? }
      format.atom { render_feed(@blogs, :title => (@project ? @project.name : Setting.app_title) + ": Blogs") }
    end
  end

  def show
    @comments = @blog.comments
    @comments.reverse! if User.current.wants_comments_in_reverse_order?
  end

  def new
    @blog = Blog.new(:project => @project, :author => User.current)
    if request.post?
      @blog.attributes = params[:blog]
      if @blog.save
        flash[:notice] = l(:notice_successful_create)
        # Mailer.deliver_blog_added(@blog) if Setting.notified_events.include?('blog_added')
        redirect_to :controller => 'blogs', :action => 'index', :project_id => @project
      end
    end
  end

  def edit
    if request.post? and @blog.update_attributes(params[:blog])
      flash[:notice] = l(:notice_successful_update)
      redirect_to :action => 'show', :id => @blog
    end
  end

  def add_comment
    @comment = Comment.new(params[:comment])
    @comment.author = User.current
    if @blog.comments << @comment
      flash[:notice] = l(:label_comment_added)
      redirect_to :action => 'show', :id => @blog
    else
      render :action => 'show'
    end
  end

  def destroy_comment
    @blog.comments.find(params[:comment_id]).destroy
    redirect_to :action => 'show', :id => @blog
  end

  def destroy
    @blog.destroy
    redirect_to :action => 'index', :project_id => @project
  end

  def preview
    @text = (params[:blog] ? params[:blog][:description] : nil)
    render :partial => 'common/preview'
  end

private
  def find_blog
    @blog = Blog.find(params[:id])
    @project = @blog.project
  rescue ActiveRecord::RecordNotFound
    render_404
  end

  def find_project
    @project = Project.find(params[:project_id])
  rescue ActiveRecord::RecordNotFound
    render_404
  end

  def find_optional_project
    return true unless params[:project_id]
    @project = Project.find(params[:project_id])
    authorize
  rescue ActiveRecord::RecordNotFound
    render_404
  end
end
