#!/bin/bash
# Run all tests for Redmine Yjs plugin
#
# This script runs:
# 1. JavaScript unit tests (Vitest)
# 2. Ruby unit/integration tests (requires Redmine checkout + PostgreSQL)
# 3. E2E tests (requires Docker Compose with Redmine + Hocuspocus)
#
# Usage:
#   ./scripts/run_all_tests.sh              # Run all tests (assumes services are running)
#   ./scripts/run_all_tests.sh --js-only    # Run only JavaScript unit tests
#   ./scripts/run_all_tests.sh --ruby-only  # Run only Ruby tests
#   ./scripts/run_all_tests.sh --e2e-only   # Run only E2E tests (assumes services are running)
#   ./scripts/run_all_tests.sh --start-services  # Start services before running tests
#   ./scripts/run_all_tests.sh --cleanup    # Cleanup Docker services only
#
# Note: For E2E tests, services must be running. Start them with:
#   ./scripts/start_test_services.sh
# Or use --start-services flag to start them automatically.
#
#!/bin/bash
set -euo pipefail
IFS=$'\n\t'

# Track child process PIDs for cleanup
CHILD_PIDS=()

# Function to kill all child processes
kill_children() {
  if [ ${#CHILD_PIDS[@]} -gt 0 ]; then
    log_warning "Killing child processes..."
    for pid in "${CHILD_PIDS[@]}"; do
      if kill -0 "$pid" 2>/dev/null; then
        kill -TERM "$pid" 2>/dev/null || true
      fi
    done
    # Wait a bit, then force kill if still running
    sleep 1
    for pid in "${CHILD_PIDS[@]}"; do
      if kill -0 "$pid" 2>/dev/null; then
        kill -KILL "$pid" 2>/dev/null || true
      fi
    done
  fi
}

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PLUGIN_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
E2E_DIR="$PLUGIN_DIR/test/e2e"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Parse arguments
JS_ONLY=false
RUBY_ONLY=false
E2E_ONLY=false
START_SERVICES=false
CLEANUP_ONLY=false
VISIBLE=false
TAGS=""

while [[ $# -gt 0 ]]; do
  case $1 in
    --js-only)
      JS_ONLY=true
      shift
      ;;
    --ruby-only)
      RUBY_ONLY=true
      shift
      ;;
    --e2e-only)
      E2E_ONLY=true
      shift
      ;;
    --start-services)
      START_SERVICES=true
      shift
      ;;
    --cleanup)
      CLEANUP_ONLY=true
      shift
      ;;
    --visible)
      VISIBLE=true
      shift
      ;;
    --tags)
      TAGS="$2"
      shift 2
      ;;
    *)
      echo "Unknown option: $1"
      echo "Usage: $0 [--js-only] [--ruby-only] [--e2e-only] [--start-services] [--cleanup] [--visible] [--tags @tagname]"
      exit 1
      ;;
  esac
done

# Helper functions
log_info() {
  echo -e "${BLUE}ℹ${NC} $1"
}

log_success() {
  echo -e "${GREEN}✓${NC} $1"
}

log_warning() {
  echo -e "${YELLOW}⚠${NC} $1"
}

log_error() {
  echo -e "${RED}✗${NC} $1"
}

log_section() {
  echo ""
  echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
  echo -e "${BLUE}$1${NC}"
  echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
  echo ""
}

# Cleanup function
cleanup_docker() {
  log_section "Cleaning up Docker services"
  cd "$E2E_DIR"
  if [ -f "docker-compose.yml" ]; then
    docker compose down -v --remove-orphans 2>/dev/null || true
    log_success "Docker services cleaned up"
  else
    log_warning "docker-compose.yml not found, skipping cleanup"
  fi
}

# Check prerequisites
check_prerequisites() {
  log_section "Checking prerequisites"
  
  # Check Docker
  if ! command -v docker &> /dev/null; then
    log_error "Docker is not installed"
    exit 1
  fi
  
  if ! docker info &> /dev/null; then
    log_error "Docker is not running"
    exit 1
  fi
  log_success "Docker is available"
  
  # Check docker compose
  if ! docker compose version &> /dev/null; then
    log_error "docker compose is not available"
    exit 1
  fi
  log_success "docker compose is available"
  
  # Check Node.js/npm for JS and E2E tests
  if [ "$E2E_ONLY" = false ] && [ "$RUBY_ONLY" = false ] && [ "$JS_ONLY" = false ]; then
    if ! command -v npm &> /dev/null; then
      log_error "npm is not installed (required for JS and E2E tests)"
      log_info "Install Node.js which includes npm: https://nodejs.org/"
      exit 1
    fi
    log_success "npm is available"
  elif [ "$JS_ONLY" = true ] || [ "$E2E_ONLY" = true ]; then
    if ! command -v npm &> /dev/null; then
      log_error "npm is not installed"
      log_info "Install Node.js which includes npm: https://nodejs.org/"
      exit 1
    fi
    log_success "npm is available"
  fi
  
  # Check Ruby for Ruby tests
  if [ "$E2E_ONLY" = false ] && [ "$RUBY_ONLY" = false ]; then
    if ! command -v ruby &> /dev/null; then
      log_warning "Ruby is not installed (Ruby tests will be skipped)"
    else
      log_success "Ruby is available"
    fi
  fi
}

# Run JavaScript unit tests
run_js_tests() {
  log_section "Running JavaScript Unit Tests"
  
  cd "$PLUGIN_DIR"
  
  # Check if package.json exists
  if [ ! -f "package.json" ]; then
    log_error "package.json not found in $PLUGIN_DIR"
    return 1
  fi
  
  # Install dependencies if needed
  if [ ! -d "node_modules" ]; then
    log_info "Installing npm dependencies..."
    if command -v npm &> /dev/null; then
      npm install || {
        log_error "Failed to install npm dependencies"
        return 1
      }
    else
      log_error "npm is not available"
      return 1
    fi
  fi
  
  # Run tests
  log_info "Running JavaScript unit tests..."
  if npm test; then
    log_success "JavaScript unit tests passed"
    return 0
  else
    log_error "JavaScript unit tests failed"
    return 1
  fi
}

# Run Ruby tests
run_ruby_tests() {
  log_section "Running Ruby Tests"
  
  # Note: Docker containers are configured for production (E2E tests), not test environment
  # Ruby tests require test database which isn't configured in production containers
  # So we skip Ruby tests when running in Docker and only run them locally
  E2E_CONTAINER="redmine_yjs_test_app"
  MAIN_CONTAINER="redmine"
  if docker ps --format '{{.Names}}' | grep -qE "^(${E2E_CONTAINER}|${MAIN_CONTAINER})$"; then
    log_info "Ruby tests skipped (Docker containers are configured for production, not test)"
    log_info "To run Ruby tests, use a local Redmine checkout"
  fi
  
  # Fallback: Check if Redmine checkout exists locally
  REDMINE_DIR="${REDMINE_DIR:-../redmine}"
  if [ -d "$REDMINE_DIR" ]; then
    log_info "Running Ruby tests against local Redmine checkout"
    
    # Check if plugin is symlinked
    PLUGIN_LINK="$REDMINE_DIR/plugins/redmine_yjs"
    if [ ! -L "$PLUGIN_LINK" ] && [ ! -d "$PLUGIN_LINK" ]; then
      log_info "Creating symlink: ln -s $PLUGIN_DIR $PLUGIN_LINK"
      mkdir -p "$REDMINE_DIR/plugins"
      ln -sf "$PLUGIN_DIR" "$PLUGIN_LINK"
      log_success "Symlink created"
    fi
    
    cd "$REDMINE_DIR"
    
    # Check database.yml
    if [ ! -f "config/database.yml" ]; then
      log_info "Creating database.yml for testing..."
      mkdir -p config
      cat > config/database.yml << EOF
test:
  adapter: postgresql
  database: redmine_test
  host: localhost
  username: redmine
  password: redmine
  encoding: utf8
EOF
      log_success "database.yml created"
    fi
    
    # Install dependencies if needed
    if [ ! -d "vendor/bundle" ]; then
      log_info "Installing Ruby dependencies..."
      bundle config set --local without 'development'
      bundle install --jobs 4 --retry 3 || {
        log_error "Failed to install Ruby dependencies"
        return 1
      }
    fi
    
    # Setup database
    log_info "Setting up database..."
    RAILS_ENV=test bundle exec rake db:create db:migrate || {
      log_error "Failed to setup database"
      return 1
    }
    
    # Run plugin migrations
    log_info "Running plugin migrations..."
    RAILS_ENV=test bundle exec rake redmine:plugins:migrate || {
      log_error "Failed to run plugin migrations"
      return 1
    }
    
    # Run tests
    log_info "Running Ruby tests..."
    if RAILS_ENV=test bundle exec rake test TEST="plugins/redmine_yjs/test/**/*_test.rb" TESTOPTS="--verbose"; then
      log_success "Ruby tests passed"
      return 0
    else
      log_error "Ruby tests failed"
      return 1
    fi
  fi
  
  # No Docker container or local checkout found
  if [ "$RUBY_ONLY" = true ]; then
    log_error "Cannot run Ruby tests:"
    log_error "  - No Docker container found (${E2E_CONTAINER} or ${MAIN_CONTAINER})"
    log_error "  - No local Redmine checkout found at ${REDMINE_DIR}"
    log_info ""
    log_info "To run Ruby tests:"
    log_info "  1. Start Docker services: ./scripts/start_test_services.sh"
    log_info "  2. Or checkout Redmine locally: git clone https://github.com/redmine/redmine.git ${REDMINE_DIR}"
    return 1
  else
    # In Docker setups, Ruby tests are optional - E2E tests cover the functionality
    log_info "Ruby tests skipped (no Docker container or local Redmine checkout found)"
    return 0
  fi
}

# Run E2E tests
run_e2e_tests() {
  log_section "Running E2E Tests"
  
  cd "$E2E_DIR"
  
  # Check if package.json exists
  if [ ! -f "package.json" ]; then
    log_error "package.json not found in $E2E_DIR"
    return 1
  fi
  
  # Install dependencies if needed (use npx npm to avoid requiring global npm)
  if [ ! -d "node_modules" ]; then
    log_info "Installing npm dependencies..."
    if command -v npm &> /dev/null; then
      npm install || {
        log_error "Failed to install npm dependencies"
        return 1
      }
    else
      log_error "npm is not available"
      return 1
    fi
  fi
  
  # Install Playwright browsers if needed (using npx)
  if ! npx --yes playwright install --check chromium &> /dev/null 2>&1; then
    log_info "Installing Playwright browsers..."
    npx --yes playwright install chromium --with-deps || {
      log_error "Failed to install Playwright browsers"
      return 1
    }
  fi
  
  # Start Docker services if requested
  if [ "$START_SERVICES" = true ]; then
    log_info "Starting Docker services using start_test_services.sh..."
    "$SCRIPT_DIR/start_test_services.sh" || {
      log_error "Failed to start Docker services"
      return 1
    }
    DOCKER_STARTED=true  # Mark that we started services
  else
    log_info "Assuming Docker services are already running"
    log_info "If services are not running, start them with: ./scripts/start_test_services.sh"
    log_info "Or use --start-services flag to start them automatically"
  fi
  
  if [ "$VISIBLE" = true ]; then
    export HEADLESS=false
    export SLOW_MO=200
  fi
  
  # Create reports directory
  mkdir -p reports/screenshots
  
  # Run tests (npm test will use npx via test-runner.sh)
  log_info "Running E2E tests..."
  log_info "Press Ctrl-C to stop (will cleanup Docker services)"
  
  # Run the test command - signals should be forwarded to it
  # Use 'set +e' temporarily so we can capture exit code
  set +e
  if [ -n "$TAGS" ]; then
    npm test -- --tags "$TAGS"
  else
    npm test
  fi
  TEST_EXIT_CODE=$?
  set -e
  
  if [ $TEST_EXIT_CODE -eq 0 ]; then
    log_success "E2E tests passed"
    return 0
  else
    log_error "E2E tests failed (exit code: $TEST_EXIT_CODE)"
    return $TEST_EXIT_CODE
  fi
}

# Main execution
main() {
  echo ""
  echo -e "${GREEN}╔═══════════════════════════════════════════════════════════╗${NC}"
  echo -e "${GREEN}║${NC}     Redmine Yjs Plugin - Test Runner                  ${GREEN}║${NC}"
  echo -e "${GREEN}╚═══════════════════════════════════════════════════════════╝${NC}"
  echo ""
  
  # Cleanup only mode
  if [ "$CLEANUP_ONLY" = true ]; then
    cleanup_docker
    exit 0
  fi
  
  # Check prerequisites
  check_prerequisites
  
  # Track results
  JS_RESULT=0
  RUBY_RESULT=0
  E2E_RESULT=0
  
  # Run JavaScript unit tests
  if [ "$RUBY_ONLY" = false ] && [ "$E2E_ONLY" = false ]; then
    if run_js_tests; then
      JS_RESULT=0
    else
      JS_RESULT=1
      if [ "$JS_ONLY" = true ]; then
        exit $JS_RESULT
      fi
    fi
  elif [ "$JS_ONLY" = true ]; then
    if run_js_tests; then
      JS_RESULT=0
    else
      JS_RESULT=1
    fi
    exit $JS_RESULT
  fi
  
  # Run Ruby tests
  if [ "$E2E_ONLY" = false ] && [ "$JS_ONLY" = false ]; then
    if run_ruby_tests; then
      RUBY_RESULT=0
    else
      RUBY_RESULT=$?
      # If Ruby tests were skipped (exit code 1 from warning), treat as success
      # Only fail if it's a real failure (exit code > 1) or if ruby-only mode
      if [ "$RUBY_ONLY" = true ]; then
        exit $RUBY_RESULT
      elif [ $RUBY_RESULT -eq 1 ]; then
        # Likely a skip (warning), treat as success for mixed test runs
        RUBY_RESULT=0
      fi
    fi
  fi
  
  # Run E2E tests
  if [ "$RUBY_ONLY" = false ] && [ "$JS_ONLY" = false ]; then
    if run_e2e_tests; then
      E2E_RESULT=0
    else
      E2E_RESULT=1
    fi
  fi
  
  # Summary
  log_section "Test Summary"
  
  if [ "$E2E_ONLY" = false ] && [ "$RUBY_ONLY" = false ]; then
    if [ $JS_RESULT -eq 0 ]; then
      log_success "JavaScript unit tests: PASSED"
    else
      log_error "JavaScript unit tests: FAILED"
    fi
  fi
  
  if [ "$E2E_ONLY" = false ] && [ "$JS_ONLY" = false ]; then
    if [ $RUBY_RESULT -eq 0 ]; then
      log_success "Ruby tests: PASSED"
    else
      log_error "Ruby tests: FAILED"
    fi
  fi
  
  if [ "$RUBY_ONLY" = false ] && [ "$JS_ONLY" = false ]; then
    if [ $E2E_RESULT -eq 0 ]; then
      log_success "E2E tests: PASSED"
    else
      log_error "E2E tests: FAILED"
    fi
  fi
  
  # Note: Cleanup is handled by trap, but only if DOCKER_STARTED=true
  # We don't need to cleanup here explicitly since the trap will handle it
  
  # Exit with appropriate code
  if [ $JS_RESULT -eq 0 ] && [ $RUBY_RESULT -eq 0 ] && [ $E2E_RESULT -eq 0 ]; then
    log_success "All tests passed!"
    exit 0
  else
    log_error "Some tests failed"
    exit 1
  fi
}

# Track if we started Docker services
DOCKER_STARTED=false

# Cleanup function that checks if we should cleanup
safe_cleanup() {
  # Kill any child processes first
  kill_children
  
  # Only cleanup if we started the services and not in cleanup-only mode
  if [ "$DOCKER_STARTED" = true ] && [ "$CLEANUP_ONLY" = false ]; then
    cleanup_docker
  fi
}

# Trap to ensure cleanup on exit (but only if we started services)
# Use a separate handler for INT/TERM that exits after cleanup
handle_interrupt() {
  echo ""
  log_warning "Interrupted by user (Ctrl-C)"
  # Kill any child processes (like npm/cucumber-js)
  kill_children
  # Cleanup Docker
  safe_cleanup
  exit 130  # Standard exit code for SIGINT
}

# Set up traps - EXIT runs on normal exit, INT/TERM on Ctrl-C/kill
trap safe_cleanup EXIT
trap handle_interrupt INT TERM

# Run main function
main

