# redMine - project management software
# Copyright (C) 2006  Jean-Philippe Lang
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

require 'ldap'
require 'iconv'

# Contributed by mathew <meta@pobox.com>
# Alternative AuthSourceLdap which uses Ruby's interface to native OpenLDAP.
# I found this more reliable than net/ldap (pure Ruby LDAP).
# OpenLDAP doesn't appear to support authentication in order to connect to
# the LDAP server, or if it does I can't find any documentation about it.
# Since the LDAP server I care about doesn't require auth, that's fine with me;
# I just note it here as an FYI.
# There's also a start_tls parameter to LDAP::SSLConn which I don't know what
# to do with. For me, both options fail, so I tunnel my LDAP instead of 
# using SSLConn.
class AuthSourceLdap < AuthSource 
  validates_presence_of :host, :port, :attr_login
  validates_presence_of :attr_firstname, :attr_lastname, :attr_mail, :if => Proc.new { |a| a.onthefly_register? }
  
  def after_initialize
    self.port = 389 if self.port == 0
  end
  
  def authenticate(login, password)
    login.downcase!
    logger.debug "replacement LDAP auth called" if logger && logger.debug?
    return nil if login.blank? || password.blank?
    attrs = []
    # Get user's DN
    ldap_con = initialize_ldap_con()
    filter = "(&(objectClass=person)(#{self.attr_login}=#{login}))"
    logger.debug "filter = #{filter}" if logger && logger.debug?
    logger.debug "base dn = #{self.base_dn}" if logger && logger.debug?
    dn = String.new
    # ruby-ldap appears to give you the dn whether you ask for it or not,
    # but via a special get_dn method rather than as an attribute.
    attrlist = [self.attr_firstname,self.attr_lastname,self.attr_mail]
    logger.debug "SEARCH" if logger && logger.debug?
    ldap_con.search(self.base_dn, LDAP::LDAP_SCOPE_SUBTREE, filter, attrlist) do |entry|
      logger.debug "search_ext returned >= 1 result" if logger && logger.debug?
      dn = entry.get_dn
      attrs = [:firstname => AuthSourceLdap.get_attr(entry, self.attr_firstname),
               :lastname => AuthSourceLdap.get_attr(entry, self.attr_lastname),
               :mail => AuthSourceLdap.get_attr(entry, self.attr_mail),
               :auth_source_id => self.id ] if onthefly_register?
    end
    logger.debug "Passed search_ext block" if logger && logger.debug?
    return nil if dn.empty?
    logger.debug "DN found for #{login}: #{dn}" if logger && logger.debug?
    # Found user, so try to authenticate
    ldap_con.unbind
    if ldap_con.bind(dn,password)
    # Return user's attributes
    logger.debug "Authentication successful for '#{login}'" if logger && logger.debug?
      return attrs    
    end
  rescue LDAP::ResultError => text
    logger.info "Authentication failure: #{text}" if logger && logger.info?
    return nil
  end

  # test the connection to the LDAP
  def test_connection
    ldap_con = initialize_ldap_con
  rescue Net::LDAP::LdapError => text
    raise "LdapError: " + text
  end
 
  def auth_method_name
    "LDAP"
  end
  
private
  def initialize_ldap_con
    logger.debug "Setting up LDAP connection to #{self.host}:#{self.port}"
    if self.tls
      # Last parameter determines if START_TLS is used for the SSL connection.
      return LDAP::SSLConn.new(self.host, self.port, true)
    else
      return LDAP::Conn.new(self.host, self.port)
    end
  end
  
  def self.get_attr(entry, attr_name)
    entry[attr_name].is_a?(Array) ? entry[attr_name].first : entry[attr_name]
  end
end
