require "application_helper"

module CustomFieldsPatch
  module IssuesHelperPatch
    def self.included(base)
      base.send(:include, InstanceMethods)
      base.class_eval do
        alias_method_chain :show_detail, :users
        alias_method_chain :render_custom_fields_rows, :users
      end
    end

    module InstanceMethods
      def render_custom_fields_rows_with_users(issue)
        return if issue.custom_field_values.empty?
        ordered_values = []
        half = (issue.custom_field_values.size / 2.0).ceil
        half.times do |i|
          ordered_values << issue.custom_field_values[i]
          ordered_values << issue.custom_field_values[i + half]
        end
        s = "<tr>\n"
        n = 0
        ordered_values.compact.each do |value|
          s << "</tr>\n<tr>\n" if n > 0 && (n % 2) == 0
          s << "\t<th>#{ h(value.custom_field.name) }:</th><td>#{value.custom_field.field_format == "user_list" ? show_value(value) :simple_format_without_paragraph(h(show_value(value))) }</td>\n"

          n += 1
        end
        s << "</tr>\n"
        s
      end

      def show_detail_with_users(detail, no_html=false)
        case detail.property
          when 'attr'
            field = detail.prop_key.to_s.gsub(/\_id$/, "")
            label = l(("field_" + field).to_sym)
            case
              when ['due_date', 'start_date'].include?(detail.prop_key)
                value = format_date(detail.value.to_date) if detail.value
                old_value = format_date(detail.old_value.to_date) if detail.old_value

              when ['project_id', 'status_id', 'tracker_id', 'assigned_to_id', 'priority_id', 'category_id', 'fixed_version_id'].include?(detail.prop_key)
                value = find_name_by_reflection(field, detail.value)
                old_value = find_name_by_reflection(field, detail.old_value)

              when detail.prop_key == 'estimated_hours'
                value = "%0.02f" % detail.value.to_f unless detail.value.blank?
                old_value = "%0.02f" % detail.old_value.to_f unless detail.old_value.blank?

              when detail.prop_key == 'parent_id'
                label = l(:field_parent_issue)
                value = "##{detail.value}" unless detail.value.blank?
                old_value = "##{detail.old_value}" unless detail.old_value.blank?
            end
          when 'cf'
            custom_field = CustomField.find_by_id(detail.prop_key)
            if custom_field
              label = custom_field.name
              value = format_value(detail.value, custom_field.field_format) if detail.value
              old_value = format_value(detail.old_value, custom_field.field_format) if detail.old_value
            end
          when 'attachment'
            label = l(:label_attachment)
        end
        call_hook(:helper_issues_show_detail_after_setting, {:detail => detail, :label => label, :value => value, :old_value => old_value})

        label ||= detail.prop_key
        value ||= detail.value
        old_value ||= detail.old_value

        unless no_html
          label = content_tag('strong', label)

          if !custom_field.nil? && custom_field.field_format == 'user_list'
            old_value = content_tag("i", old_value) if detail.old_value
          else
            old_value = content_tag("i", h(old_value)) if detail.old_value
          end

          old_value = content_tag("strike", old_value) if detail.old_value and (!detail.value or detail.value.empty?)
          if detail.property == 'attachment' && !value.blank? && a = Attachment.find_by_id(detail.prop_key)
            # Link to the attachment if it has not been removed
            value = link_to_attachment(a)
          elsif !custom_field.nil? && custom_field.field_format == 'user_list'
            value = content_tag("i", value) if value
          else
            value = content_tag("i", h(value)) if value
          end
        end

        if !detail.value.blank?
          case detail.property
            when 'attr', 'cf'
              if !detail.old_value.blank?
                l(:text_journal_changed, :label => label, :old => old_value, :new => value)
              else
                l(:text_journal_set_to, :label => label, :value => value)
              end
            when 'attachment'
              l(:text_journal_added, :label => label, :value => value)
          end
        else
          l(:text_journal_deleted, :label => label, :old => old_value)
        end
      end
    end
  end

  module CustomFieldsHelperPatch
    def self.included(base)
      base.send(:include, InstanceMethods)
      base.class_eval do
        alias_method_chain :custom_field_tag, :users
        alias_method_chain :format_value, :users
      end
    end

    module InstanceMethods

      def format_value_with_users(value, field_format)
        return (value.blank? ? '-' : link_to_user(User.find(:first, :conditions => ["login = ?", value]))) if field_format=='user_list'
        Redmine::CustomFieldFormat.format_value(value, field_format) # Proxy
      end

      def custom_field_tag_with_users(name, custom_value)
        custom_field = custom_value.custom_field
        field_name = "#{name}[custom_field_values][#{custom_field.id}]"
        field_id = "#{name}_custom_field_values_#{custom_field.id}"

        field_format = Redmine::CustomFieldFormat.find_by_name(custom_field.field_format)
        case field_format.edit_as
          when "date"
            text_field_tag(field_name, custom_value.value, :id => field_id, :size => 10) +
                    calendar_for(field_id)
          when "text"
            text_area_tag(field_name, custom_value.value, :id => field_id, :rows => 3, :style => 'width:90%')
          when "bool"
            hidden_field_tag(field_name, '0') + check_box_tag(field_name, '1', custom_value.true?, :id => field_id)
          when "list"
            blank_option = custom_field.is_required? ?
                    (custom_field.default_value.blank? ? "<option value=\"\">--- #{l(:actionview_instancetag_blank_option)} ---</option>" : '') :
                    '<option></option>'
            select_tag(field_name, blank_option + options_for_select(custom_field.possible_values, custom_value.value), :id => field_id)
          when "user_list"
            users = @issue == nil ? User.find(:all).sort : @issue.assignable_users
            users.delete_if { |user| user.anonymous? }
            custom_field.possible_values.clear()
            users.each { |user| custom_field.possible_values << user unless user.anonymous? }

            blank_option = custom_field.is_required? ?
                    (custom_field.default_value.blank? ? "<option value=\"\">--- #{l(:actionview_instancetag_blank_option)} ---</option>" : '') :
                    '<option></option>'
            select_tag(field_name, blank_option + options_for_select(users.collect { |x| [x.name, x.login] }, custom_value.value), :id => field_id)
          else
#            field_format.edit_as
            text_field_tag(field_name, custom_value.value, :id => field_id)
        end
      end
    end
  end

  module CustomFieldFormatPatch
    # include ApplicationHelper

    def self.included(base)
#      base.send(:include, InstanceMethods)
    end

    module InstanceMethods
      def format_as_user_list(value)
        # value.blank? ? '-' : link_to_user(User.find(:first, :conditions => ["login = ?", value]))
        value
      end
    end
  end

  module CustomFieldPatch
    def self.included(base)
      base.send(:include, InstanceMethods)
      base.class_eval do
        alias_method_chain :order_statement, :users
        alias_method_chain :cast_value, :users
      end
    end

    module InstanceMethods

      def cast_value_with_users(value)
        casted = cast_value_without_users(value)
        if (casted == nil && !value.blank? && field_format=='user_list')
          casted = User.find(:first, :conditions => ["login = ?", value])
        end
        casted  
      end

      def order_statement_with_users
        case field_format
          when 'string', 'text', 'list', 'date', 'bool', 'user_list'
            # COALESCE is here to make sure that blank and NULL values are sorted equally
            "COALESCE((SELECT cv_sort.value FROM #{CustomValue.table_name} cv_sort" +
                    " WHERE cv_sort.customized_type='#{self.class.customized_class.name}'" +
                    " AND cv_sort.customized_id=#{self.class.customized_class.table_name}.id" +
                    " AND cv_sort.custom_field_id=#{id} LIMIT 1), '')"
          when 'int', 'float'
            # Make the database cast values into numeric
            # Postgresql will raise an error if a value can not be casted!
            # CustomValue validations should ensure that it doesn't occur
            "(SELECT CAST(cv_sort.value AS decimal(60,3)) FROM #{CustomValue.table_name} cv_sort" +
                    " WHERE cv_sort.customized_type='#{self.class.customized_class.name}'" +
                    " AND cv_sort.customized_id=#{self.class.customized_class.table_name}.id" +
                    " AND cv_sort.custom_field_id=#{id} AND cv_sort.value <> '' AND cv_sort.value IS NOT NULL LIMIT 1)"
          else
            nil
        end
      end
    end
  end

  module IssuePatch
    def self.included(base)
      base.send(:include, InstanceMethods)
      base.class_eval do
        alias_method_chain :recipients, :custom_field_users
      end
    end
  end

  module InstanceMethods
    def recipients_with_custom_field_users
      mails = recipients_without_custom_field_users
      users = []
      available_custom_fields().each { |x|
        users << User.find(:first, :conditions => ["login = ?", x.value]) if x.type=='user_list'
      }
      users.uniq!
      # Remove users that can not view the issue
      users.reject! { |user| !visible?(user) }
      mails << users.collect(&:mail)
      mails
    end
  end

  module QueryPatch

    def self.included(base)
      base.send(:include, InstanceMethods)
      base.class_eval do
        alias_method_chain :add_custom_fields_filters, :users
      end
    end

    module InstanceMethods
      def add_custom_fields_filters_with_users(custom_fields)
        @available_filters ||= {}
        users = @project.assignable_users
        custom_fields.select(&:is_filter?).each do |field|
          case field.field_format
            when "text"
              options = {:type => :text, :order => 20}
            when "list"
              options = {:type => :list_optional, :values => field.possible_values, :order => 20}
            when "user_list"
              options = {:type => :list_optional, :values => users.collect { |x| [x.name, x.login] }, :order => 20}
            when "date"
              options = {:type => :date, :order => 20}
            when "bool"
              options = {:type => :list, :values => [[l(:general_text_yes), "1"], [l(:general_text_no), "0"]], :order => 20}
            else
              options = {:type => :string, :order => 20}
          end
          @available_filters["cf_#{field.id}"] = options.merge({:name => field.name})
        end
      end
    end
  end
end