# Redmine - project management software
# Copyright (C) 2006-2011  Jean-Philippe Lang
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

# Portions of this code adapted from Michael Vance, see:
#  http://www.redmine.org/issues/339
# Adapted by Terry Suereth.

require 'redmine/scm/adapters/perforce_adapter'

class Repository::Perforce < Repository
  validates_presence_of :root_url, :url

  def self.scm_adapter_class
    Redmine::Scm::Adapters::PerforceAdapter
  end

  def self.scm_name
    'Perforce'
  end

  def supports_directory_revisions?
    false
  end

  def repo_log_encoding
    'UTF-8'
  end

  def latest_changesets(path, rev, limit=10)
    revisions = scm.revisions(path, rev, nil, :limit => limit)
    revisions ? changesets.find_all_by_revision(revisions.collect(&:identifier), :order => "committed_on DESC", :include => :user) : []
  end

  def fetch_changesets
    scm_info = scm.info
    if scm_info
      # latest revision found in database
      db_revision = latest_changeset ? latest_changeset.revision.to_i : 0
      # latest revision in the repository
      scm_revision = scm_info.lastrev.identifier.to_i
      if db_revision < scm_revision
        logger.debug "Fetching changesets for repository #{url}" if logger && logger.debug?
        identifier_from = db_revision + 1
        while (identifier_from <= scm_revision)
          # loads changesets by batches of 200
          identifier_to = [identifier_from + 199, scm_revision].min
          revisions = scm.revisions('', identifier_to, identifier_from, :with_paths => true)
          revisions.reverse_each do |revision|
            transaction do
              changeset = Changeset.create(:repository   => self,
                                           :revision     => revision.identifier,
                                           :committer    => revision.author,
                                           :committed_on => revision.time,
                                           :comments     => revision.message)

              revision.paths.each do |change|
                changeset.create_change(change)
              end unless changeset.new_record?
            end
          end unless revisions.nil?
          identifier_from = identifier_to + 1
        end
      end
    end
  end

end
