# redMine - project management software
# Copyright (C) 2006-2007  Jean-Philippe Lang
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

require 'redmine/scm/adapters/perforce_adapter'

class Repository::Perforce < Repository
  
  def scm_adapter
    Redmine::Scm::Adapters::PerforceAdapter
  end
  
  def self.scm_name
    'Perforce'
  end

  def supports_annotate?
    # not sure on this one?
    false
  end

  def changesets_for_path(path)
    revisions = scm.revisions(path)
    revisions ? changesets.find_all_by_revision(revisions.collect(&:identifier), :order => "committed_on DESC") : []
  end
  
  # Returns a path relative to the url of the repository
  def relative_path(path)
    scm.relative_path(path)
  end

  def fetch_changesets
    scm_info = scm.info
    if scm_info
      # latest revision found in database
      db_revision = latest_changeset ? latest_changeset.revision.to_i : 0
      # latest revision in the repository
      scm_revision = scm_info.lastrev.identifier.to_i
      if db_revision < scm_revision
        logger.debug "Fetching changesets for repository #{url}" if logger && logger.debug?
        identifier_from = db_revision + 1
        while (identifier_from <= scm_revision)
          # loads changesets by batches of 200
          identifier_to = [identifier_from + 199, scm_revision].min
          revisions = scm.revisions('', identifier_to, identifier_from, :with_paths => true)
          transaction do
            revisions.reverse_each do
              |revision|
              changeset = Changeset.create(:repository => self,
                :revision => revision.identifier,
                :committer => revision.author,
                :committed_on => revision.time,
                :comments => revision.message)
              
              revision.paths.each do
                |change|

                revpath = '/' + scm.relative_path( change.depot_file )
                action = ( change.action )
                revaction = case action
                  when "add" then "A"
                  when "delete" then "D"
                  when "edit" then "M"
                  when "branch" then "B"
                  #when "integrate" then "I"
                  #when "import" then "E"
                end

                Change.create(:changeset => changeset,
                  :action => revaction,
                  :path =>  revpath,
                  :revision => '#' + ( change.revno.to_s + '/' + change.head.to_s ))
              end
            end
          end unless revisions.nil?
          identifier_from = identifier_to + 1
        end
      end
    end
  end
  
  private
  
  # Returns the relative url of the repository
  # Eg: root_url = file:///var/svn/foo
  #     url      = file:///var/svn/foo/bar
  #     => returns /bar
  #def relative_url
    #@relative_url ||= url.gsub(Regexp.new("^#{Regexp.escape(root_url)}"), '')
  #  @relative_url = (url.end_with?("/") ? url : url + "/")
  #end
end
