# frozen_string_literal: true

# Redmine - project management software
# Copyright (C) 2006-2019  Jean-Philippe Lang
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

class MemberImport < Import
  def self.menu_item
    :members
  end

  def self.authorized?(user)
    user.allowed_to?(:import_members, nil, :global => true)
  end

  # Returns the objects that were imported
  def saved_objects
    object_ids = saved_items.pluck(:obj_id)
    objects = Member.where(:id => object_ids).order(:id).preload(:user, :roles)
  end

  def mappable_custom_fields_for_ss
    # there is no feature to make custom field for Member
    []
  end

  def allowed_target_projects
    Project.allowed_to(user, :manage_members).order(:lft)
  end

  def project
    project_id = mapping["project_id"].to_i
    allowed_target_projects.find_by_id(project_id) || allowed_target_projects.first
  end

  def mappable_core_fields_for_ss
    ["exclude", "user", "role"]
  end

  # concatenates core fields and custom fields as field names
  def mappable_fields_for_ss
    mappable_fields = self.mappable_core_fields_for_ss
    mappable_fields
  end

  # makes local language version of fields list for select
  def attribute_options_for_ss
    attributes = []
    core_fields = self.mappable_core_fields_for_ss
    core_fields.each do |field|
      field_symbol_str = "field_" + field.sub(/_ids?\z/, "")
      local_attribute_name = l(field_symbol_str.to_sym)
      if local_attribute_name
        attributes << local_attribute_name
      end
    end

    i = -1
    attributes.map { |h| [h, i += 1] }
  end

  # validates each row of CSV/SS by tentatively build_object
  def validate_objects_for_ss
    self.settings["errors"] = {}
    self.settings["ignores"] = {}
    self.settings["excludes"] = {}

    read_items do |row, position|
      errors_per_position = {}
      ignores_per_position = {}

      if exclude = row_value(row, "exclude")
        self.settings["excludes"].merge!({ position => true })
        next
      end
      ## Since Member class does not have safe_attributes= method, moving values directly.
      member = Member.new
      ## Project: mandatory
      member.project_id = mapping["project_id"].to_i

      unless member.project.id == mapping["project_id"].to_i
        errors_per_position.merge!({ "project_id" => mapping["project_id"] })
      end
      ## user_id: mandatory. so far Group is not importable.
      if user_name = row_value(row, "user")
        if member_user = Principal.detect_by_keyword(Principal.all.visible, user_name)
          member.user_id = member_user.id
          ## if a Group name is given, the member does not have user.
          unless member.user && member.user.name == row_value(row, "user")
            errors_per_position.merge!({ "user" => mapping["user"] })
          end
          if Project.find_by_id(member.project_id).users.include?(member_user)
            errors_per_position.merge!({ "user" => mapping["user"] })
          end
        else
          errors_per_position.merge!({ "user" => mapping["user"] })
        end
      end
      ## roles: mandatory. 1 or more role is applicable to a member.
      if raw_role_names = row_value(row, "role")
        role_names = raw_role_names.split(",")
        role_names.each do |role_name|
          role_name.sub!(/\A[[:blank:]]*/, "")
          role_name.sub!(/[[:blank:]]*\z/, "")
        end
        role_ids = []
        role_names.each do |role_name|
          if role = Role.find_by_name(role_name)
            role_ids << role.id
          else
            errors_per_position.merge!({ "role" => mapping["role"] })
          end
        end
        member.role_ids = role_ids

        unless member.validate
          errors_per_position.merge!({ "role" => mapping["role"] })
        end
      else
        ## user is entered but role isn't
        errors_per_position.merge!({ "role" => mapping["role"] })
      end

      ## finally, merge errors and ignores per line into all errors
      if errors_per_position.count > 0
        self.settings["errors"].merge!({ position => errors_per_position })
      else
        self.settings["errors"].delete(position)
      end
      if ignores_per_position.count > 0
        self.settings["ignores"].merge!({ position => ignores_per_position })
      else
        self.settings["ignores"].delete(position)
      end
    end
  end

  private

  def build_object(row, item)
    object = Member.new()

    object.project_id = mapping["project_id"]

    if member_name = row_value(row, "user")
      if member_user = Principal.detect_by_keyword(Principal.all.visible, member_name)
        object.user_id = member_user.id
      end
    end
    if raw_role_names = row_value(row, "role")
      role_names = raw_role_names.split(",")
      role_names.each do |role_name|
        role_name.sub!(/\A[[:blank:]]*/, "")
        role_name.sub!(/[[:blank:]]*\z/, "")
      end
      role_ids = []
      role_names.each do |role_name|
        if role = Role.find_by_name(role_name)
          role_ids << role.id
        end
      end
      object.role_ids = role_ids
    end

    object
  end
end
