# a patch against Import model
ActiveSupport::Reloader.to_prepare do
  unless Import.included_modules.include?(SsImport::ImportPatch)
    Import.send(:include, SsImport::ImportPatch)
  end
end

require_dependency "import"
require "roo"
require "redmine"

module SsImport
  module ImportPatch
    def self.included(base) # :nodoc:
      base.send(:include, InstanceMethods)
      base.include(ClassMethods)

      base.class_eval do
        unloadable
        alias_method :file_without_ss=, :file=
        alias_method :file=, :file_with_ss=

        alias_method :set_default_settings_without_ss, :set_default_settings
        alias_method :set_default_settings, :set_default_settings_with_ss

        alias_method :read_rows_without_ss, :read_rows
        alias_method :read_rows, :read_rows_with_ss

        alias_method :run_without_ss, :run
        alias_method :run, :run_with_ss
      end
    end

    module InstanceMethods
      # Copied from import.rb to avoid uninitialized constant error.
      DATE_FORMATS = [
        "%Y-%m-%d",
        "%d/%m/%Y",
        "%m/%d/%Y",
        "%Y/%m/%d",
        "%d.%m.%Y",
        "%d-%m-%Y",
      ]

      # Override so that original_filename's extention is saved in settings.
      def file_with_ss=(arg)
        return unless arg.present? && arg.size > 0
        basename, extension = /\A(.+?)((?:\.[^.]+)?)\z/.match(arg.original_filename, &:captures)
        self.filename = generate_filename
        Redmine::Utils.save_upload(arg, filepath)
        self.settings.merge!(
          "extension" => extension.gsub(".", ""),
        )
      end

      # override to extract spreadsheet's sheets
      def set_default_settings_with_ss(options = {})
        separator = lu(user, :general_csv_separator)
        if file_exists?
          begin
            content = File.read(filepath, 256)
            separator = [",", ";"].sort_by { |sep| content.count(sep) }.last
          rescue => e
          end
        end
        wrapper = '"'
        encoding = lu(user, :general_csv_encoding)

        date_format = lu(user, "date.formats.default", :default => "foo")
        date_format = DATE_FORMATS.first unless DATE_FORMATS.include?(date_format)

        self.settings.merge!(
          "separator" => separator,
          "wrapper" => wrapper,
          "encoding" => encoding,
          "date_format" => date_format,
          "notifications" => "0",
        )

        if options.key?(:project_id) && !options[:project_id].blank?
          # Do not fail if project doesn't exist
          begin
            project = Project.find(options[:project_id])
            self.settings.merge!("mapping" => { "project_id" => project.id })
          rescue
          end
        else
          # In case when project is not in the context, defaults to first allowed one.
          begin
            project = Project.allowed_to(user, :import_issues).first
            self.settings.merge!("mapping" => { "project_id" => project.id })
          rescue
          end
        end
        # this fetch is just for sheets extraction. data is not used.
        if filepath
          data = Roo::Spreadsheet.open(filepath, { extension: self.settings["extension"] })
          datatype = data.class.to_s
          sheets = data.sheets
          self.settings.merge!(
            "datatype" => datatype,
            "sheets" => sheets,
          )
        end
        # adding update import feature
        self.settings.merge!("import_type" => "create")
      end

      # Override with roo. roo reads CSV as well.
      def read_rows_with_ss
        return unless file_exists?
        # originally exist settings for CSV
        if @matrix == nil
          csv_options = { :headers => false }
          csv_options[:encoding] = settings["encoding"].to_s.presence || "UTF-8"
          csv_options[:encoding] = "bom|UTF-8" if csv_options[:encoding] == "UTF-8"
          separator = settings["separator"].to_s
          csv_options[:col_sep] = separator if separator.size == 1
          wrapper = settings["wrapper"].to_s
          csv_options[:quote_char] = wrapper if wrapper.size == 1
          # settings for roo
          roo_options = { :extension => settings["extension"].to_s }
          roo_options[:csv_options] = csv_options

          spreadsheet = Roo::Spreadsheet.open(filepath, roo_options)

          if settings["datatype"] == "Roo::CSV"
            @matrix = spreadsheet.to_matrix.to_a
          else
            sheet_selected = settings["sheets"]
            @matrix = spreadsheet.sheet(sheet_selected).to_matrix.to_a
          end
        end

        @matrix.each do |row|
          yield row if block_given?
        end
      end

      # Overrides to make the notification bulk
      # this patch depends on another plugin "bulk_notification"
      def run_with_ss(options = {})
        max_items = options[:max_items]
        max_time = options[:max_time]
        current = 0
        imported = 0
        resume_after = items.maximum(:position) || 0
        interrupted = false
        started_on = Time.now

        read_items do |row, position|
          if (max_items && imported >= max_items) || (max_time && Time.now >= started_on + max_time)
            interrupted = true
            break
          end
          if position > resume_after
            item = items.build
            item.position = position
            item.unique_id = row_value(row, "unique_id") if use_unique_id?

            exclude = row_value(row, "exclude")

            unless exclude
              if object = build_object(row, item)
                if object.save
                  item.obj_id = object.id
                  build_relations_for_ss(row, item) if object.class == Issue
                else
                  item.message = object.errors.full_messages.join("\n")
                end
              end

              item.save!
              imported += 1

              do_callbacks(use_unique_id? ? item.unique_id : item.position, object)
            else # excluded
              item.message = l(:text_excluded)
            end
          end
          current = position
        end

        if imported == 0 || interrupted == false
          if total_items.nil?
            update_attribute :total_items, current
          end
          ## ss_import addition
          if settings["notifications"] == "1"
            issues = []
            journals = []
            items.each do |item|
              if item.obj_id
                issue = Issue.find_by_id(item.obj_id)
                issues << issue
                journals << issue.journals.last
              end
            end
            case settings["import_type"]
            when "create"
              BulkNotificationMailer.deliver_issue_add_bulk(issues)
            when "update"
              BulkNotificationMailer.deliver_issue_edit_bulk(journals)
            else
              ## do nothing
            end
          end
          ## end of ss_import addition
          update_attribute :finished, true
          remove_file
        end

        current
      end

      def create?
        "create" == settings["import_type"]
      end

      def update?
        "update" == settings["import_type"]
      end
    end

    module ClassMethods
    end
  end
end
