# redMine - project management software
# Copyright (C) 2006-2007  Jean-Philippe Lang
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

class MAPIMailHandler
  include ActionView::Helpers::SanitizeHelper

  class UnauthorizedAction < StandardError; end
  class MissingInformation < StandardError; end
  
  attr_reader :email, :user
  
  def self.receive(email, options={})
    @@handler_options = options.dup
    @@handler_options[:issue] ||= {}
  
    @@handler_options[:allow_override] = @@handler_options[:allow_override].split(',').collect(&:strip) if @@handler_options[:allow_override].is_a?(String)
    @@handler_options[:allow_override] ||= []

    # Project needs to be overridable if not specified
    @@handler_options[:allow_override] << 'project' unless @@handler_options[:issue].has_key?(:project)
    # Status overridable by default
    @@handler_options[:allow_override] << 'status' unless @@handler_options[:issue].has_key?(:status)    

    #super email
	new.receive(email)
  end
  
  # Processes incoming emails
  def receive(email)
    @email = email
    @user = User.active.find(:first, :conditions => ["LOWER(mail) = ?", email.sender.address.strip.downcase])
    unless @user
      # Unknown user => the email is ignored
      # TODO: ability to create the user's account
	  puts "email submitted by unknown user #{email.sender.address}"
      logger.info "MailHandler: email submitted by unknown user [#{email.sender.address}]" if logger && logger.info
      return false
    end
	puts "user identified #{@user}"
    User.current = @user
    dispatch
  end
  
  private
  
  def logger
        RAILS_DEFAULT_LOGGER
  end

  ISSUE_REPLY_SUBJECT_RE = %r{\[[^\]]+#(\d+)\]}
  
  def dispatch
    if m = email.subject.match(ISSUE_REPLY_SUBJECT_RE)
      receive_issue_update(m[1].to_i)
    else
      receive_issue
    end
  rescue ActiveRecord::RecordInvalid => e
    # TODO: send a email to the user
    logger.error e.message if logger
    false
  rescue MissingInformation => e
    logger.error "MailHandler: missing information from #{user}: #{e.message}" if logger
    false
  rescue UnauthorizedAction => e
    logger.error "MailHandler: unauthorized attempt from #{user}" if logger
    false
  end
  
  # Creates a new issue
  def receive_issue
    project = target_project
    tracker = (get_keyword(:tracker) && project.trackers.find_by_name(get_keyword(:tracker))) || project.trackers.find(:first)
    category = (get_keyword(:category) && project.issue_categories.find_by_name(get_keyword(:category)))
    priority = (get_keyword(:priority) && Enumeration.find_by_opt_and_name('IPRI', get_keyword(:priority)))
    status =  (get_keyword(:status) && IssueStatus.find_by_name(get_keyword(:status)))

	puts "receive_issue [#{target_project}]"
    # check permission
    raise UnauthorizedAction unless user.allowed_to?(:add_issues, project)
    issue = Issue.new(:author => user, :project => project, :tracker => tracker, :category => category, :priority => priority)
    # check workflow
    if status && issue.new_statuses_allowed_to(user).include?(status)
      issue.status = status
    end
    issue.subject = email.subject.chomp.toutf8
    issue.description = plain_text_body
    # custom fields
    issue.custom_field_values = issue.available_custom_fields.inject({}) do |h, c|
      if value = get_keyword(c.name, :override => true)
        h[c.id] = value
      end
      h
    end
    issue.save!
    add_attachments(issue)
    logger.info "MailHandler: issue ##{issue.id} created by #{user}" if logger && logger.info
    # add To and Cc as watchers
    add_watchers(issue)
    # send notification after adding watchers so that they can reply to Redmine
    Mailer.deliver_issue_add(issue) if Setting.notified_events.include?('issue_added')
    issue
  end
  
  def target_project
    # TODO: other ways to specify project:
    # * parse the email To field
    # * specific project (eg. Setting.mail_handler_target_project)
    target = Project.find_by_identifier(get_keyword(:project))
    raise MissingInformation.new('Unable to determine target project') if target.nil?
    target
  end
  
  # Adds a note to an existing issue
  def receive_issue_update(issue_id)
    status =  (get_keyword(:status) && IssueStatus.find_by_name(get_keyword(:status)))
    
    issue = Issue.find_by_id(issue_id)
    return unless issue
    # check permission
    raise UnauthorizedAction unless user.allowed_to?(:add_issue_notes, issue.project) || user.allowed_to?(:edit_issues, issue.project)
    raise UnauthorizedAction unless status.nil? || user.allowed_to?(:edit_issues, issue.project)

    # add the note
    journal = issue.init_journal(user, plain_text_body)
    add_attachments(issue)
    # check workflow
    if status && issue.new_statuses_allowed_to(user).include?(status)
      issue.status = status
    end
    issue.save!
    logger.info "MailHandler: issue ##{issue.id} updated by #{user}" if logger && logger.info
    Mailer.deliver_issue_edit(journal) if Setting.notified_events.include?('issue_updated')
    journal
  end
  
  def add_attachments(obj)
    #if email.has_attachments?
      email.attachments.each do |attachment|
		puts "attachment :#{attachment.name}"
		attachment.fields.each do |field|
			puts "Field: name=#{field.name} type=#{field.type} value=#{field.value} index=#{field.index} id=#{field.id}"
		end
		attachment.ole_methods.each do |method|
			puts "Method: name=#{method.name}"
		end
		
		if attachment.Type == 1 #|| attachment.Type == 3
			filename = "c:\\tmp\\#{attachment.name}"
			puts "#{filename} source #{attachment.source.size}"
			attachment.WriteToFile( filename );
			Attachment.create(:container => obj,
                          :file => File.new(filename),
                          :author => user,
                          :content_type => attachment.fields.item(15))
			File.delete(filename)
		end
	  end
    #end
  end
  
  # Adds To and Cc as watchers of the given object if the sender has the
  # appropriate permission
  def add_watchers(obj)
    if user.allowed_to?("add_#{obj.class.name.underscore}_watchers".to_sym, obj.project)
		addresses = []
		email.recepients.each do |recipient|
			if reciepient.type < 3
				addresses += recipient.address
			end
		end
      #addresses = [email.to, email.cc].flatten.compact.uniq.collect {|a| a.strip.downcase}
      unless addresses.empty?
        watchers = User.active.find(:all, :conditions => ['LOWER(mail) IN (?)', addresses])
        watchers.each {|w| obj.add_watcher(w)}
      end
    end
  end
  
  def get_keyword(attr, options={})
    @keywords ||= {}
    if @keywords.has_key?(attr)
      @keywords[attr]
    else
      @keywords[attr] = begin
        if (options[:override] || @@handler_options[:allow_override].include?(attr.to_s)) && plain_text_body.gsub!(/^#{attr}[ \t]*:[ \t]*(.+)\s*$/i, '')
          $1.strip
        elsif !@@handler_options[:issue][attr].blank?
          @@handler_options[:issue][attr]
        end
      end
    end
  end
  
  # Returns the text/plain part of the email
  # If not found (eg. HTML-only email), returns the body with tags removed
  def plain_text_body
    return @plain_text_body unless @plain_text_body.nil?
	
    #parts = @email.parts.collect {|c| (c.respond_to?(:parts) && !c.parts.empty?) ? c.parts : c}.flatten
    #if parts.empty?
    #  parts << @email
    #end
    #plain_text_part = parts.detect {|p| p.content_type == 'text/plain'}
	plain_text_part = email.text
    if plain_text_part.nil?
      # no text/plain part found, assuming html-only email
      # strip html tags and remove doctype directive
      @plain_text_body = strip_tags(@email.text)
      @plain_text_body.gsub! %r{^<!DOCTYPE .*$}, ''
    else
      @plain_text_body = plain_text_part
    end
    @plain_text_body.strip!
    @plain_text_body
  end
end
