# Redmine - project management software
# Copyright (C) 2006-2008  Jean-Philippe Lang
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

module Redmine
  module Helpers
    # Simple class to handle gantt chart data
    class Gantt
      include ERB::Util
      include Redmine::I18n

      # :nodoc:
      # Some utility methods for the PDF export
      class PDF
        MaxCharactorsForSubject = 45
        TotalWidth = 280
        LeftPaneWidth = 100

        def self.right_pane_width
          TotalWidth - LeftPaneWidth
        end
      end

      attr_reader :year_from, :month_from, :date_from, :date_to, :zoom, :months, :truncated, :max_rows
      attr_accessor :query
      attr_accessor :project
      attr_accessor :view

      def initialize(options={})
        options = options.dup

        if options[:year] && options[:year].to_i >0
          @year_from = options[:year].to_i
          if options[:month] && options[:month].to_i >=1 && options[:month].to_i <= 12
            @month_from = options[:month].to_i
          else
            @month_from = 1
          end
        else
          @month_from ||= Date.today.month
          @year_from ||= Date.today.year
        end

        zoom = (options[:zoom] || User.current.pref[:gantt_zoom]).to_i
        @zoom = (zoom > 0 && zoom < 5) ? zoom : 2    
        months = (options[:months] || User.current.pref[:gantt_months]).to_i
        @months = (months > 0 && months < 25) ? months : 6

        # Save gantt parameters as user preference (zoom and months count)
        if (User.current.logged? && (@zoom != User.current.pref[:gantt_zoom] || @months != User.current.pref[:gantt_months]))
          User.current.pref[:gantt_zoom], User.current.pref[:gantt_months] = @zoom, @months
          User.current.preference.save
        end

        @date_from = Date.civil(@year_from, @month_from, 1)
        @date_to = (@date_from >> @months) - 1
        
        @subjects = ''
        @lines = ''
        @number_of_rows = nil
        
        @issue_ancestors = []
        
        @truncated = false
        if options.has_key?(:max_rows)
          @max_rows = options[:max_rows]
        else
          @max_rows = Setting.gantt_items_limit.blank? ? nil : Setting.gantt_items_limit.to_i
        end
      end

      def common_params
        { :controller => 'gantts', :action => 'show', :project_id => @project }
      end
      
      def params
        common_params.merge({  :zoom => zoom, :year => year_from, :month => month_from, :months => months })
      end
      
      def params_previous
        common_params.merge({:year => (date_from << months).year, :month => (date_from << months).month, :zoom => zoom, :months => months })
      end
      
      def params_next
        common_params.merge({:year => (date_from >> months).year, :month => (date_from >> months).month, :zoom => zoom, :months => months })
      end

      ### Extracted from the HTML view/helpers
      # Returns the number of rows that will be rendered on the Gantt chart
      def number_of_rows
        @number_of_rows
      end

      # Renders the subjects of the Gantt chart, the left side.
      def subjects(options={})
        render(options.merge(:only => :subjects)) unless @subjects_rendered
        @subjects
      end

      # Renders the lines of the Gantt chart, the right side
      def lines(options={})
        render(options.merge(:only => :lines)) unless @lines_rendered
        @lines
      end

      def render(options={})
        options = {:indent => 4, :render => :subject, :format => :html}.merge(options)

        @subjects = '' unless options[:only] == :lines
        @lines = '' unless options[:only] == :subjects
        @number_of_rows = 0

        options[:render_issues_without_version_first] = true unless options.key? :render_issues_without_version_first

        options[:top] = 0 unless options.key? :top
        options[:indent_increment] = 20 unless options.key? :indent_increment
        options[:top_increment] = 20 unless options.key? :top_increment

        options[:gantt_indent] = options[:indent]

        # We sort by project.lft so that projects are listed before their children
        top_projects = Project.find( :all,
          :conditions => @query.project_statement(Project.table_name),
          :order => ["#{Project.table_name}.lft ASC"])

        # Purge descendants to be sure to have only top projects left
        prev_top = nil
        top_projects.delete_if do |top|
          remove_descendant = prev_top && top.is_descendant_of?(prev_top)
          prev_top = top
          remove_descendant
        end

        # Get issues with theirs versions and subprojects as filtered by query criterias
        issues = Issue.find(
          :all,
          :include => [:tracker, :status, :assigned_to, :priority, :project, :fixed_version],
          :conditions => @query.statement,
          :joins => [
            "LEFT OUTER JOIN #{Project.table_name} AS top_project ON top_project.id IN (#{top_projects.collect(&:id).join(', ')}) AND top_project.lft <= #{Project.table_name}.lft AND top_project.rgt >= #{Project.table_name}.rgt " + # Top project
            "LEFT OUTER JOIN #{Project.table_name} AS version_project ON version_project.id = #{Version.table_name}.project_id " + # Project of the fixed version
            "LEFT OUTER JOIN #{Project.table_name} AS version_top_project ON " +
                    "(version_top_project.id = #{Project.table_name}.id AND #{Version.table_name}.sharing = 'none')" +
                " OR (version_top_project.id = top_project.id AND #{Version.table_name}.sharing IN ('hierarchy', 'system', 'tree'))" +
                " OR (version_top_project.id = CASE WHEN top_project.lft < version_project.lft THEN version_project.id ELSE top_project.id END AND #{Version.table_name}.sharing = 'descendants')"], 
          :order => [
            "top_project.lft ASC, top_project.name ASC, " + # Top project
            "COALESCE(version_top_project.lft, #{Project.table_name}.lft) ASC, " + # Version top project or project if none
            "CASE WHEN #{Issue.table_name}.fixed_version_id IS NULL THEN #{ options[:render_issues_without_version_first] ? '0 ELSE 1' : '1 ELSE 0' } END ASC, " + # Issues without version first (or last)
            "CASE WHEN #{Version.table_name}.effective_date IS NULL THEN 0 ELSE 1 END DESC, " + # Version with effective_date first
            "#{Version.table_name}.effective_date ASC, #{Version.table_name}.name ASC, " +
            "#{Project.table_name}.lft ASC"] )

        # TODO : at this point we could have been returned 3 extra Project objects useful for building project path : top_project, version_project and top_version_project
        # There is no way ActiveRecord can map sql query results to more than one model (hibernate does), so we have to compute them again in update_and_render_project_path(), adding a dummy time & cpu overhead to the whole rendering process
        # Idea : use an IssueForGantt class with association to issue, top_project, version_project and top_version_project ?

        prev_issue = nil
        path_issues = []

        # TODO : use a Hash rather than an Array when switching to Ruby 1.9, as Hash is ordered in 1.9
        issue_path = []

        # Render Gantt by looping on issues
        issues.each do |i|

          # Stack issues within the same project path to allow them to be sorted before rendering
          if prev_issue && (
                prev_issue.project != i.project ||
                prev_issue.fixed_version != i.fixed_version)

            render_path_issues(path_issues, options) unless path_issues.empty?
            break if abort?

            path_issues.clear
          end

          issue_path = update_and_render_project_path(top_projects, i, prev_issue, issue_path, options)
          break if abort?

          path_issues << i

          prev_issue = i
        end

        render_path_issues(path_issues, options) unless abort?

        @subjects_rendered = true unless options[:only] == :lines
        @lines_rendered = true unless options[:only] == :subjects

        render_end(options)
      end

      # Renders issues in a version/project path
      def render_path_issues(issues, options)

        sort_issues! issues

        base_indent = options[:indent]

        prev_issue = nil
        issue_hierarchy = []

        issues.each do |i|

          # Indent issue hierarchy (issue in the same hierarchy tree should follow thanks to the sorting order)
          ancestor_idx = -1
          (issue_hierarchy.length - 1).downto(0) do |h_i|
            if issue_hierarchy[h_i].is_ancestor_of?(i)
              ancestor_idx = h_i
              break
            end
          end
          issue_hierarchy.slice!((ancestor_idx + 1)..-1)

          issue_hierarchy << i

          options[:indent] = base_indent + (issue_hierarchy.length - 1) * options[:indent_increment]

          # Render issue node
          subject_for_issue(i, options) unless options[:only] == :lines
          line_for_issue(i, options) unless options[:only] == :subjects
          @number_of_rows += 1
          break if abort?
          options[:top] += options[:top_increment]

          prev_issue = i
        end

      end

      def update_and_render_project_path(top_projects, i, prev_issue, issue_path, options)

        prev_version = prev_issue.nil? ? nil : prev_issue.fixed_version
        project_change = prev_issue.nil? || i.project != prev_issue.project

        first_item_to_render_idx = nil

        # Render the project path to the project the issue belongs to
        if project_change

          top_project = issue_path.empty? ? nil : issue_path.first 

          # Remove invalid path queue
          if top_project.nil? || !i.project.is_or_is_descendant_of?(top_project)
            # New branch
            top_project = top_projects.detect { |p| p.is_or_is_ancestor_of?(i.project) }
            top_projects.delete(top_project) # Lighten the top_projects map as a top project should not be rendered twice 
            issue_path = [top_project]
            first_item_to_render_idx = 0
            prev_version = nil
          elsif !prev_issue.nil? && !i.project.is_descendant_of?(prev_issue.project)
            # Truncate actual path (in any way, we keep the top project node)
            # TODO : use index { } with Ruby 1.8.7+
            first_distinct_project_idx = nil
            1.upto(issue_path.length - 1) do |pi_idx|
              if issue_path[pi_idx].is_a?(Project) && !issue_path[pi_idx].is_ancestor_of?(i.project)
                first_distinct_project_idx = pi_idx
                break
              end
            end
            issue_path.slice!(first_distinct_project_idx..-1)
          end

          # TODO : use rindex { } with Ruby 1.9
          last_common_project_idx = nil
          (issue_path.length - 1).downto(0) do |pi_idx|
            if issue_path[pi_idx].is_a?(Project)
              last_common_project_idx = pi_idx
              break
            end
          end

          # Complete path
          rpath_queue = []
          subproject = i.project
          until subproject == issue_path.at(last_common_project_idx)
            rpath_queue << subproject
            subproject = subproject.parent
          end

          unless rpath_queue.empty?
            first_item_to_render_idx = issue_path.length if first_item_to_render_idx.nil?
            issue_path += rpath_queue.reverse 
          end
        end

        version_change = i.fixed_version != prev_version

        if version_change
          # Remove previous Version node from path
          unless prev_version.nil?
            prev_version_idx = issue_path.index(prev_version)
            unless prev_version_idx.nil? # May have been removed with the project path truncation
              issue_path.delete_at(prev_version_idx)
              first_item_to_render_idx = prev_version_idx if prev_version_idx < issue_path.length || (!first_item_to_render_idx.nil? && prev_version_idx < first_item_to_render_idx)
            end
          end

          # Place new Version node within the path
          unless i.fixed_version.nil?
            # TODO : use index { } with Ruby 1.8.7+
            new_version_idx = nil
            0.upto(issue_path.length - 1) do |pi_idx|
              # TODO : p.shared_versions() can slow down rendering, maybe add a has_shared_version?(v) in Project ?
              if issue_path[pi_idx].shared_versions.include?(i.fixed_version)
                new_version_idx = pi_idx + 1
                break
              end
            end
            issue_path.insert(new_version_idx, i.fixed_version)
            first_item_to_render_idx = new_version_idx if first_item_to_render_idx.nil? || new_version_idx < first_item_to_render_idx
          end
        end

        # Render new path part
        unless first_item_to_render_idx.nil?
          options[:indent] = options[:gantt_indent] + first_item_to_render_idx * options[:indent_increment]

          issue_path.slice(first_item_to_render_idx..-1).each do |pi|

            if pi.is_a?(Version)
              # Render version node
              render_version(pi, options)
            else
              # Render project node
              render_subproject(pi, options)
            end
            break if abort?
            options[:indent] += options[:indent_increment]
            options[:top] += options[:top_increment]
          end
        else
          options[:indent] = options[:gantt_indent] + issue_path.length * options[:indent_increment]
        end

        issue_path
      end

      def render_version(version, options={})
        # Version header
        subject_for_version(version, options) unless options[:only] == :lines
        line_for_version(version, options) unless options[:only] == :subjects
        @number_of_rows += 1
      end

      def render_subproject(subproject, options={})
        # Subproject header
        subject_for_project(subproject, options) unless options[:only] == :lines
        line_for_project(subproject, options) unless options[:only] == :subjects
        @number_of_rows += 1
      end

      def render_end(options={})
        case options[:format]
        when :pdf        
          options[:pdf].Line(15, options[:top], PDF::TotalWidth, options[:top])
        end
      end

      def subject_for_project(project, options)
        case options[:format]
        when :html
          subject = "<span class='icon icon-projects #{project.overdue? ? 'project-overdue' : ''}'>"
          subject << view.link_to_project(project)
          subject << '</span>'
          html_subject(options, subject, :css => "project-name")
        when :image
          image_subject(options, project.name)
        when :pdf
          pdf_new_page?(options)
          pdf_subject(options, project.name)
        end
      end

      def line_for_project(project, options)
        # Skip versions that don't have a start_date or due date
        if project.is_a?(Project) && project.start_date && project.due_date
          options[:zoom] ||= 1
          options[:g_width] ||= (self.date_to - self.date_from + 1) * options[:zoom]
            
          coords = coordinates(project.start_date, project.due_date, nil, options[:zoom])
          label = h(project)
          
          case options[:format]
          when :html
            html_task(options, coords, :css => "project task", :label => label, :markers => true)
          when :image
            image_task(options, coords, :label => label, :markers => true, :height => 3)
          when :pdf
            pdf_task(options, coords, :label => label, :markers => true, :height => 0.8)
          end
        else
          ActiveRecord::Base.logger.debug "Gantt#line_for_project was not given a project with a start_date"
          ''
        end
      end

      def subject_for_version(version, options)
        case options[:format]
        when :html
          subject = "<span class='icon icon-package #{version.behind_schedule? ? 'version-behind-schedule' : ''} #{version.overdue? ? 'version-overdue' : ''}'>"
          subject << view.link_to_version(version)
          subject << '</span>'
          html_subject(options, subject, :css => "version-name")
        when :image
          image_subject(options, version.to_s_with_project)
        when :pdf
          pdf_new_page?(options)
          pdf_subject(options, version.to_s_with_project)
        end
      end

      def line_for_version(version, options)
        # Skip versions that don't have a start_date
        if version.is_a?(Version) && version.start_date && version.due_date
          options[:zoom] ||= 1
          options[:g_width] ||= (self.date_to - self.date_from + 1) * options[:zoom]

          coords = coordinates(version.start_date, version.due_date, version.completed_pourcent, options[:zoom])
          label = "#{h version } #{h version.completed_pourcent.to_i.to_s}%"
          label = h("#{version.project} -") + label unless @project && @project == version.project

          case options[:format]
          when :html
            html_task(options, coords, :css => "version task", :label => label, :markers => true)
          when :image
            image_task(options, coords, :label => label, :markers => true, :height => 3)
          when :pdf
            pdf_task(options, coords, :label => label, :markers => true, :height => 0.8)
          end
        else
          ActiveRecord::Base.logger.debug "Gantt#line_for_version was not given a version with a start_date"
          ''
        end
      end

      def subject_for_issue(issue, options)
        output = case options[:format]
        when :html
          css_classes = ''
          css_classes << ' issue-overdue' if issue.overdue?
          css_classes << ' issue-behind-schedule' if issue.behind_schedule?
          css_classes << ' icon icon-issue' unless Setting.gravatar_enabled? && issue.assigned_to

          subject = "<span class='#{css_classes}'>"
          if issue.assigned_to.present?
            assigned_string = l(:field_assigned_to) + ": " + issue.assigned_to.name
            subject << view.avatar(issue.assigned_to, :class => 'gravatar icon-gravatar', :size => 10, :title => assigned_string).to_s
          end
          subject << view.link_to_issue(issue)
          subject << '</span>'
          html_subject(options, subject, :css => "issue-subject", :title => issue.subject) + "\n"
        when :image
          image_subject(options, issue.subject)
        when :pdf
          pdf_new_page?(options)
          pdf_subject(options, issue.subject)
        end

        output
      end

      def line_for_issue(issue, options)
        # Skip issues that don't have a due_before (due_date or version's due_date)
        if issue.is_a?(Issue) && issue.due_before
          coords = coordinates(issue.start_date, issue.due_before, issue.done_ratio, options[:zoom])
          label = "#{ issue.status.name } #{ issue.done_ratio }%"
          
          case options[:format]
          when :html
            html_task(options, coords, :css => "task " + (issue.leaf? ? 'leaf' : 'parent'), :label => label, :issue => issue, :markers => !issue.leaf?)
          when :image
            image_task(options, coords, :label => label)
          when :pdf
            pdf_task(options, coords, :label => label)
        end
        else
          ActiveRecord::Base.logger.debug "GanttHelper#line_for_issue was not given an issue with a due_before"
          ''
        end
      end

      # Generates a gantt image
      # Only defined if RMagick is avalaible
      def to_image(format='PNG')
        date_to = (@date_from >> @months)-1    
        show_weeks = @zoom > 1
        show_days = @zoom > 2
        
        subject_width = 400
        header_heigth = 18
        # width of one day in pixels
        zoom = @zoom*2
        g_width = (@date_to - @date_from + 1)*zoom
        g_height = 20 * 5 + 30
        headers_heigth = (show_weeks ? 2*header_heigth : header_heigth)
        height = g_height + headers_heigth
        width = subject_width+g_width+1

        imgl = Magick::ImageList.new
        imgl.new_image(width, height)
        gc = Magick::Draw.new

        # Subjects
        gc.stroke('transparent')
        subjects(:image => gc, :top => (headers_heigth + 20), :indent => 4, :format => :image)
        g_height = 20 * number_of_rows + 30
        height = g_height + headers_heigth
        imgl.last.resize!(width, height)

        # Months headers
        month_f = @date_from
        left = subject_width
        @months.times do 
          width = ((month_f >> 1) - month_f) * zoom
          gc.fill('white')
          gc.stroke('grey')
          gc.stroke_width(1)
          gc.rectangle(left, 0, left + width, height)
          gc.fill('black')
          gc.stroke('transparent')
          gc.stroke_width(1)
          gc.text(left.round + 8, 14, "#{month_f.year}-#{month_f.month}")
          left = left + width
          month_f = month_f >> 1
        end

        # Weeks headers
        if show_weeks
        	left = subject_width
        	height = header_heigth
        	if @date_from.cwday == 1
        	    # date_from is monday
                week_f = date_from
        	else
        	    # find next monday after date_from
        		week_f = @date_from + (7 - @date_from.cwday + 1)
        		width = (7 - @date_from.cwday + 1) * zoom
                gc.fill('white')
                gc.stroke('grey')
                gc.stroke_width(1)
                gc.rectangle(left, header_heigth, left + width, 2*header_heigth + g_height-1)
        		left = left + width
        	end
        	while week_f <= date_to
        		width = (week_f + 6 <= date_to) ? 7 * zoom : (date_to - week_f + 1) * zoom
                gc.fill('white')
                gc.stroke('grey')
                gc.stroke_width(1)
                gc.rectangle(left.round, header_heigth, left.round + width, 2*header_heigth + g_height-1)
                gc.fill('black')
                gc.stroke('transparent')
                gc.stroke_width(1)
                gc.text(left.round + 2, header_heigth + 14, week_f.cweek.to_s)
        		left = left + width
        		week_f = week_f+7
        	end
        end

        # Days details (week-end in grey)
        if show_days
        	left = subject_width
        	height = g_height + header_heigth - 1
        	wday = @date_from.cwday
        	(date_to - @date_from + 1).to_i.times do 
              width =  zoom
              gc.fill(wday == 6 || wday == 7 ? '#eee' : 'white')
              gc.stroke('#ddd')
              gc.stroke_width(1)
              gc.rectangle(left, 2*header_heigth, left + width, 2*header_heigth + g_height-1)
              left = left + width
              wday = wday + 1
              wday = 1 if wday > 7
        	end
        end

        # border
        gc.fill('transparent')
        gc.stroke('grey')
        gc.stroke_width(1)
        gc.rectangle(0, 0, subject_width+g_width, headers_heigth)
        gc.stroke('black')
        gc.rectangle(0, 0, subject_width+g_width, g_height+ headers_heigth-1)

        # content
        top = headers_heigth + 20

        gc.stroke('transparent')
        lines(:image => gc, :top => top, :zoom => zoom, :subject_width => subject_width, :format => :image)
        
        # today red line
        if Date.today >= @date_from and Date.today <= date_to
          gc.stroke('red')
          x = (Date.today-@date_from+1)*zoom + subject_width
          gc.line(x, headers_heigth, x, headers_heigth + g_height-1)      
        end

        gc.draw(imgl)
        imgl.format = format
        imgl.to_blob
      end if Object.const_defined?(:Magick)

      def to_pdf
        pdf = ::Redmine::Export::PDF::IFPDF.new(current_language)
        pdf.SetTitle("#{l(:label_gantt)} #{project}")
        pdf.AliasNbPages
        pdf.footer_date = format_date(Date.today)
        pdf.AddPage("L")
        pdf.SetFontStyle('B',12)
        pdf.SetX(15)
        pdf.Cell(PDF::LeftPaneWidth, 20, project.to_s)
        pdf.Ln
        pdf.SetFontStyle('B',9)

        subject_width = PDF::LeftPaneWidth
        header_heigth = 5

        headers_heigth = header_heigth
        show_weeks = false
        show_days = false

        if self.months < 7
          show_weeks = true
          headers_heigth = 2*header_heigth
          if self.months < 3
            show_days = true
            headers_heigth = 3*header_heigth
          end
        end

        g_width = PDF.right_pane_width
        zoom = (g_width) / (self.date_to - self.date_from + 1)
        g_height = 120
        t_height = g_height + headers_heigth

        y_start = pdf.GetY

        # Months headers
        month_f = self.date_from
        left = subject_width
        height = header_heigth
        self.months.times do 
          width = ((month_f >> 1) - month_f) * zoom 
          pdf.SetY(y_start)
          pdf.SetX(left)
          pdf.Cell(width, height, "#{month_f.year}-#{month_f.month}", "LTR", 0, "C")
          left = left + width
          month_f = month_f >> 1
        end  

        # Weeks headers
        if show_weeks
          left = subject_width
          height = header_heigth
          if self.date_from.cwday == 1
            # self.date_from is monday
            week_f = self.date_from
          else
            # find next monday after self.date_from
            week_f = self.date_from + (7 - self.date_from.cwday + 1)
            width = (7 - self.date_from.cwday + 1) * zoom-1
            pdf.SetY(y_start + header_heigth)
            pdf.SetX(left)
            pdf.Cell(width + 1, height, "", "LTR")
            left = left + width+1
          end
          while week_f <= self.date_to
            width = (week_f + 6 <= self.date_to) ? 7 * zoom : (self.date_to - week_f + 1) * zoom
            pdf.SetY(y_start + header_heigth)
            pdf.SetX(left)
            pdf.Cell(width, height, (width >= 5 ? week_f.cweek.to_s : ""), "LTR", 0, "C")
            left = left + width
            week_f = week_f+7
          end
        end

        # Days headers
        if show_days
          left = subject_width
          height = header_heigth
          wday = self.date_from.cwday
          pdf.SetFontStyle('B',7)
          (self.date_to - self.date_from + 1).to_i.times do 
            width = zoom
            pdf.SetY(y_start + 2 * header_heigth)
            pdf.SetX(left)
            pdf.Cell(width, height, day_name(wday).first, "LTR", 0, "C")
            left = left + width
            wday = wday + 1
            wday = 1 if wday > 7
          end
        end

        pdf.SetY(y_start)
        pdf.SetX(15)
        pdf.Cell(subject_width+g_width-15, headers_heigth, "", 1)

        # Tasks
        top = headers_heigth + y_start
        options = {
          :top => top,
          :zoom => zoom,
          :subject_width => subject_width,
          :g_width => g_width,
          :indent => 0,
          :indent_increment => 5,
          :top_increment => 5,
          :format => :pdf,
          :pdf => pdf
        }
        render(options)
        pdf.Output
      end

      private

      def coordinates(start_date, end_date, progress, zoom=nil)
        zoom ||= @zoom

        coords = {}
        if start_date && end_date && start_date < self.date_to && end_date > self.date_from
          if start_date > self.date_from
            coords[:start] = start_date - self.date_from
            coords[:bar_start] = start_date - self.date_from
          else
            coords[:bar_start] = 0
          end
          if end_date < self.date_to
            coords[:end] = end_date - self.date_from
            coords[:bar_end] = end_date - self.date_from + 1
          else
            coords[:bar_end] = self.date_to - self.date_from + 1
          end

          if progress
            progress_date = start_date + (end_date - start_date) * (progress / 100.0)
            if progress_date > self.date_from && progress_date > start_date
              if progress_date < self.date_to
                coords[:bar_progress_end] = progress_date - self.date_from + 1
              else
                coords[:bar_progress_end] = self.date_to - self.date_from + 1
              end
            end

            if progress_date < Date.today
              late_date = [Date.today, end_date].min
              if late_date > self.date_from && late_date > start_date
                if late_date < self.date_to
                  coords[:bar_late_end] = late_date - self.date_from + 1
                else
                  coords[:bar_late_end] = self.date_to - self.date_from + 1
                end
              end
            end
          end
        end

        # Transforms dates into pixels witdh
        coords.keys.each do |key|
          coords[key] = (coords[key] * zoom).floor
        end
        coords
      end

      # Sorts a collection of issues for gantt rendering
      def sort_issues!(issues)
        issues.sort! { |a, b| gantt_issue_compare(a, b) }
      end

      # Compare issues for rendering order 
      def gantt_issue_compare(x, y)

        if x.parent_id == y.parent_id
          # Same level in issue hierarchy
          basic_gantt_issue_compare(x, y)
        elsif x.root_id == y.root_id || x.root_id == y.id || y.root_id == x.id
          # Same issue hierarchy
          x.lft <=> y.lft
        else
          # Distinct hierarchies
          x.root_id <=> y.root_id
          # TODO : advanced sort between issues when ancestors have been filtered out by query
        end
      end

      def basic_gantt_issue_compare(x, y)
        if x.start_date == y.start_date
          x.id <=> y.id
        elsif x.start_date.nil?
          1 # null date appears at the end
        elsif y.start_date.nil?
          -1
        else
          x.start_date <=> y.start_date
        end
      end

      def current_limit
        if @max_rows
          @max_rows - @number_of_rows
        else
          nil
        end
      end

      def abort?
        if @max_rows && @number_of_rows >= @max_rows
          @truncated = true
        end
      end

      def pdf_new_page?(options)
        if options[:top] > 180
          options[:pdf].Line(15, options[:top], PDF::TotalWidth, options[:top])
          options[:pdf].AddPage("L")
          options[:top] = 15
          options[:pdf].Line(15, options[:top] - 0.1, PDF::TotalWidth, options[:top] - 0.1)
        end
      end

      def html_subject(params, subject, options={})
        style = "position: absolute;top:#{params[:top]}px;left:#{params[:indent]}px;"
        style << "width:#{params[:subject_width] - params[:indent]}px;" if params[:subject_width]
        
        output = view.content_tag 'div', subject, :class => options[:css], :style => style, :title => options[:title]
        @subjects << output
        output
      end

      def pdf_subject(params, subject, options={})
        params[:pdf].SetY(params[:top])
        params[:pdf].SetX(15)

        char_limit = PDF::MaxCharactorsForSubject - params[:indent]
        params[:pdf].Cell(params[:subject_width]-15, 5, (" " * params[:indent]) +  subject.to_s.sub(/^(.{#{char_limit}}[^\s]*\s).*$/, '\1 (...)'), "LR")

        params[:pdf].SetY(params[:top])
        params[:pdf].SetX(params[:subject_width])
        params[:pdf].Cell(params[:g_width], 5, "", "LR")
      end

      def image_subject(params, subject, options={})
        params[:image].fill('black')
        params[:image].stroke('transparent')
        params[:image].stroke_width(1)
        params[:image].text(params[:indent], params[:top] + 2, subject)
      end

      def html_task(params, coords, options={})
        output = ''
        # Renders the task bar, with progress and late
        if coords[:bar_start] && coords[:bar_end]
          output << "<div style='top:#{ params[:top] }px;left:#{ coords[:bar_start] }px;width:#{ coords[:bar_end] - coords[:bar_start] - 2}px;' class='#{options[:css]} task_todo'>&nbsp;</div>"
          
          if coords[:bar_late_end]
            output << "<div style='top:#{ params[:top] }px;left:#{ coords[:bar_start] }px;width:#{ coords[:bar_late_end] - coords[:bar_start] - 2}px;' class='#{options[:css]} task_late'>&nbsp;</div>"
          end
          if coords[:bar_progress_end]
            output << "<div style='top:#{ params[:top] }px;left:#{ coords[:bar_start] }px;width:#{ coords[:bar_progress_end] - coords[:bar_start] - 2}px;' class='#{options[:css]} task_done'>&nbsp;</div>"
          end
        end
        # Renders the markers
        if options[:markers]
          if coords[:start]
            output << "<div style='top:#{ params[:top] }px;left:#{ coords[:start] }px;width:15px;' class='#{options[:css]} marker starting'>&nbsp;</div>"
          end
          if coords[:end]
            output << "<div style='top:#{ params[:top] }px;left:#{ coords[:end] + params[:zoom] }px;width:15px;' class='#{options[:css]} marker ending'>&nbsp;</div>"
          end
        end
        # Renders the label on the right
        if options[:label]
          output << "<div style='top:#{ params[:top] }px;left:#{ (coords[:bar_end] || 0) + 8 }px;' class='#{options[:css]} label'>"
          output << options[:label]
          output << "</div>"
        end
        # Renders the tooltip
        if options[:issue] && coords[:bar_start] && coords[:bar_end]
          output << "<div class='tooltip' style='position: absolute;top:#{ params[:top] }px;left:#{ coords[:bar_start] }px;width:#{ coords[:bar_end] - coords[:bar_start] }px;height:12px;'>"
          output << '<span class="tip">'
          output << view.render_issue_tooltip(options[:issue])
          output << "</span></div>"
        end
        @lines << output
        output
      end

      def pdf_task(params, coords, options={})
        height = options[:height] || 2
        
        # Renders the task bar, with progress and late
        if coords[:bar_start] && coords[:bar_end]
          params[:pdf].SetY(params[:top]+1.5)
          params[:pdf].SetX(params[:subject_width] + coords[:bar_start])
          params[:pdf].SetFillColor(200,200,200)
          params[:pdf].Cell(coords[:bar_end] - coords[:bar_start], height, "", 0, 0, "", 1)
            
          if coords[:bar_late_end]
            params[:pdf].SetY(params[:top]+1.5)
            params[:pdf].SetX(params[:subject_width] + coords[:bar_start])
            params[:pdf].SetFillColor(255,100,100)
            params[:pdf].Cell(coords[:bar_late_end] - coords[:bar_start], height, "", 0, 0, "", 1)
          end
          if coords[:bar_progress_end]
            params[:pdf].SetY(params[:top]+1.5)
            params[:pdf].SetX(params[:subject_width] + coords[:bar_start])
            params[:pdf].SetFillColor(90,200,90)
            params[:pdf].Cell(coords[:bar_progress_end] - coords[:bar_start], height, "", 0, 0, "", 1)
          end
        end
        # Renders the markers
        if options[:markers]
          if coords[:start]
            params[:pdf].SetY(params[:top] + 1)
            params[:pdf].SetX(params[:subject_width] + coords[:start] - 1)
            params[:pdf].SetFillColor(50,50,200)
            params[:pdf].Cell(2, 2, "", 0, 0, "", 1) 
          end
          if coords[:end]
            params[:pdf].SetY(params[:top] + 1)
            params[:pdf].SetX(params[:subject_width] + coords[:end] - 1)
            params[:pdf].SetFillColor(50,50,200)
            params[:pdf].Cell(2, 2, "", 0, 0, "", 1) 
          end
        end
        # Renders the label on the right
        if options[:label]
          params[:pdf].SetX(params[:subject_width] + (coords[:bar_end] || 0) + 5)
          params[:pdf].Cell(30, 2, options[:label])
        end
      end

      def image_task(params, coords, options={})
        height = options[:height] || 6
        
        # Renders the task bar, with progress and late
        if coords[:bar_start] && coords[:bar_end]
          params[:image].fill('#aaa')
          params[:image].rectangle(params[:subject_width] + coords[:bar_start], params[:top], params[:subject_width] + coords[:bar_end], params[:top] - height)
 
          if coords[:bar_late_end]
            params[:image].fill('#f66')
            params[:image].rectangle(params[:subject_width] + coords[:bar_start], params[:top], params[:subject_width] + coords[:bar_late_end], params[:top] - height)
          end
          if coords[:bar_progress_end]
            params[:image].fill('#00c600')
            params[:image].rectangle(params[:subject_width] + coords[:bar_start], params[:top], params[:subject_width] + coords[:bar_progress_end], params[:top] - height)
          end
        end
        # Renders the markers
        if options[:markers]
          if coords[:start]
            x = params[:subject_width] + coords[:start]
            y = params[:top] - height / 2
            params[:image].fill('blue')
            params[:image].polygon(x-4, y, x, y-4, x+4, y, x, y+4)
          end
          if coords[:end]
            x = params[:subject_width] + coords[:end] + params[:zoom]
            y = params[:top] - height / 2
            params[:image].fill('blue')
            params[:image].polygon(x-4, y, x, y-4, x+4, y, x, y+4)
          end
        end
        # Renders the label on the right
        if options[:label]
          params[:image].fill('black')
          params[:image].text(params[:subject_width] + (coords[:bar_end] || 0) + 5,params[:top] + 1, options[:label])
        end
      end
    end
  end
end
